/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 *************************************************************************************************/
/**
 * @file Application.h
 *
 * @brief This file contains declaration which are independant from any bus system.
 * 
 * @author R. Walter (basis: E. Ott)
 *
 */

#ifndef __APPLICATION_H
#define __APPLICATION_H

#include "Common.h"

#ifdef __cplusplus
  extern "C" {
#endif  /* _cplusplus */

/**
 * @brief This structure is part of all application data.
 * It contains all common data structures independent of the stack or bus.
 *
 */
typedef struct APPLICATION_COMMON_Ttag
{
  /**
   * @brief Handle from driver api.
   */
  TLR_HANDLE hDriver;
    
  /**
   * @brief Channel handle from driver api.
   */
  TLR_HANDLE hChannel;

  /**
   * @brief Default timeout.
   */
  TLR_UINT32 ulTimeout;

  /**
   * @brief Buffer for received process data.
   */
  TLR_UINT8* pabReadBuffer;

  /**
   * @brief Size of buffer for received process data.
   */
  TLR_UINT32 ulReadBufferSize;

  /**
   * @brief Buffer for transmit process data.
   */
  TLR_UINT8* pabWriteBuffer;

  /**
   * @brief Size of buffer for transmit process data.
   */
  TLR_UINT32 ulWriteBufferSize;

} APPLICATION_COMMON_T;


/**
 * @brief An application contains at least out of the header with the common elements.
 * Thus we create a data type for such basic applications, so we can easily cast to it.
 *
 */
typedef struct APPLICATION_Ttag
{
  APPLICATION_COMMON_T tCommon;
} APPLICATION_T;


/*************************************************************************************************
 * Below you will find some common functions, 
 * they are implemented by Application.h and can be used by any application.
 */

/**
 * @brief Common helper function for sending requests and receiving confirmations.
 * Unknown packets will be forwarded to the common packet handler.
 * This function may only be used during system startup, 
 * when no indications are expected to come up. 
 * Cascading calls of this function may end up in trouble.
 *
 * @param ptApp        Pointer to application data.
 * @param ptSendPkt    Pointer to packet which should be send (Request).
 * @param ptRecvPkt    Pointer to received packet (Confirmation).
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_SendRecvPkt(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt);


/**
 * @brief Common helper function for sending empty requests (only command codes) 
 * and receiving confirmations. This packet may be used for any registration effords.
 * Unknown packets will be forwarded to the common packet handler.
 * This function may only be used during system startup, 
 * when no indications are expected to come up. 
 * Cascading calls of this function may end up in trouble.
 *
 * @param ptApp        Pointer to application data.
 * @param ulCmd        Command code which should be used for empty packet.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_SendRecvEmptyPkt(APPLICATION_T* ptApp, TLR_UINT32 ulCmd);


/**
 * @brief Main entry function which manages the startup of the application.
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Main(APPLICATION_T* ptApp);


/*************************************************************************************************
 * Below you will find some common interface functions, 
 * which must be implemented by all applications.
 */

/**
 * @brief This function builds the packet which is used for
 * configuring the stack.
 *
 * @param ptApp        Pointer to application data. 
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigurationReqV2(CIFX_PACKET* ptPacket);

/**
 * @brief This function builds the packet which is used for
 * configuring the stack.
 *
 * @param ptApp        Pointer to application data. 
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigurationReqV4(CIFX_PACKET* ptPacket);

/**
 * @brief This function creates all resources (memory allocation) of the application.
 *
 * @param pptApp        Pointer to pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_CreateResources(APPLICATION_T** pptApp);


/**
 * @brief This function frees all resources (free memory) of the application.
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_FreeResources(APPLICATION_T* ptApp);


/**
 * @brief This function prompts an intro message to the standard output at startup.
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_PromptIntro(APPLICATION_T* ptApp);


/**
 * @brief This function initializes the application.
 * It initializes the applicaton data and retrieves handles from the driver. 
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Initialize(APPLICATION_T* ptApp);


/**
 * @brief This function finalizes the application.
 * It returns handles to the driver. 
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Finalize(APPLICATION_T* ptApp);


/**
 * @brief This function is the standard packet handler of the application.
 * All received packets (during cyclic operation) will be forwarded to this function.
 *
 * @param ptApp        Pointer to application data.
 * @param ptPacket     Pointer to received packet.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandlePacket(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket);


/**
 * @brief This function handles the process data exchange (receiving and transmitting).
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandleProcessData(APPLICATION_T* ptApp);


#ifdef __cplusplus
  }
#endif  /* _cplusplus */


#endif /* #ifndef __APPLICATION_H */
